﻿#requires -version 5.1
<#
  Install-HunterUIOffline.ps1 (robust, payload-flex, PATH-less re-detect, SikuliX detect)
  Volledig offline installer (GUI + headless) voor:
   - Temurin JDK 17 (MSI)
   - .NET 9 Windows Desktop Runtime (x64, EXE)
   - SikuliX API v2.0.5 (JAR -> %ProgramData%\HunterUI\sikulix\)
   - HunterUI Application (download van https://csar.bio/HunterUI-latest.zip)

  Payload-map: .\payload\ naast dit script.
  Voorbeelden bestandsnamen:
    - OpenJDK17U-jdk_x64_windows_hotspot_17.0.16_8.msi
    - windowsdesktop-runtime-9.0.9-win-x64.exe
    - sikulixapi-2.0.5.jar of sikulixapi-2.0.5-win.jar
#>

param(
  [switch]$Headless  # zonder GUI draaien
)

# ---------- Algemene instellingen ----------
$ErrorActionPreference = 'Stop'
$ProgressPreference    = 'SilentlyContinue'

# ---------- Paden ----------
$ScriptDir = Split-Path -Parent $PSCommandPath
$Payload   = Join-Path $ScriptDir 'payload'
$RootDir   = Join-Path $env:ProgramData 'HunterUI'
$LogDir    = Join-Path $RootDir 'PrereqInstaller'
$SikuliDir = Join-Path $RootDir 'sikulix'
$SikuliJar = Join-Path $SikuliDir 'sikulixapi-2.0.5.jar'
$HunterUIDir = 'C:\Program Files\HunterUI'
$HunterUIExe = Join-Path $HunterUIDir 'HunterUI.exe'
$LogFile   = Join-Path $LogDir ("install_{0}.log" -f (Get-Date -Format 'yyyyMMdd_HHmmss'))

# ---------- Voorbereiden ----------
New-Item -ItemType Directory -Force -Path $RootDir, $LogDir | Out-Null

# ---------- Lock-vrije logging ----------
function Write-Log([string]$Message) {
  try {
    $ts   = (Get-Date).ToString('yyyy-MM-dd HH:mm:ss')
    $line = "$ts  $Message"
    Write-Host $Message
    $max=5; $i=0
    while ($true) {
      try { Add-Content -Path $LogFile -Value $line -Encoding UTF8; break }
      catch { $i++; if ($i -ge $max) { throw }; Start-Sleep -Milliseconds 120 }
    }
  } catch { Write-Host "[logfout] $($_.Exception.Message)" }
}

# ---------- UAC elevatie ----------
function Ensure-Admin {
  $isAdmin = ([Security.Principal.WindowsPrincipal][Security.Principal.WindowsIdentity]::GetCurrent()).
    IsInRole([Security.Principal.WindowsBuiltInRole]::Administrator)
  if (-not $isAdmin) {
    Write-Log "Administratorrechten vereist – herstart met UAC."
    $args = "-NoProfile -ExecutionPolicy Bypass -File `"$PSCommandPath`""
    if ($Headless) { $args += " -Headless" }
    $psi = New-Object System.Diagnostics.ProcessStartInfo "powershell.exe"
    $psi.Arguments = $args
    $psi.Verb      = "runas"
    [Diagnostics.Process]::Start($psi) | Out-Null
    exit
  }
}

# ---------- Helpers ----------
function Test-Cmd($name) { if (Get-Command $name -ErrorAction SilentlyContinue) { $true } else { $false } }

# Run native tool robust (no throw); capture stdout+stderr text
function Run-Native([string]$exe, [string]$args) {
  try { (& cmd.exe /c "$exe $args" 2>&1 | Out-String) } catch { "" }
}

# Safer existence checks that do NOT depend on PATH/env refresh
function Get-DotNetExePath {
  $p = Join-Path ${env:ProgramFiles} 'dotnet\dotnet.exe'
  if (Test-Path $p) { return $p }
  return $null
}

function Has-Java17 {
  try {
    # A) Registry probe (both 64/32 views)
    $keys = @(
      'HKLM:\SOFTWARE\Eclipse Adoptium\JDK\17',
      'HKLM:\SOFTWARE\JavaSoft\JDK\17',
      'HKLM:\SOFTWARE\Microsoft\JDK\17',
      'HKLM:\SOFTWARE\WOW6432Node\JavaSoft\JDK\17'
    )
    foreach ($k in $keys) { if (Test-Path $k) { return $true } }

    # B) Well-known install locations (no PATH needed)
    $candidates = @()
    $candidates += Get-ChildItem -Path 'C:\Program Files' -Filter 'jdk-17*' -Directory -ErrorAction SilentlyContinue | ForEach-Object { Join-Path $_.FullName 'bin\java.exe' }
    $candidates += Get-ChildItem -Path 'C:\Program Files\Eclipse Adoptium' -Filter 'jdk-17*' -Directory -ErrorAction SilentlyContinue | ForEach-Object { Join-Path $_.FullName 'bin\java.exe' }
    foreach ($j in $candidates) { if (Test-Path $j) { return $true } }

    # C) Fallback: java -version only if available on PATH
    if (Test-Cmd 'java') {
      $v = Run-Native 'java' '-version'
      if ($v -match 'version\s+"17\.') { return $true }
    }
    return $false
  } catch { return $false }
}

function Has-DotNet     { if (Get-DotNetExePath) { $true } else { $false } }
function Has-DesktopRt9 {
  try {
    $rtDir = Join-Path ${env:ProgramFiles} 'dotnet\shared\Microsoft.WindowsDesktop.App'
    if (Test-Path $rtDir) {
      $hit = Get-ChildItem -Path $rtDir -Directory -ErrorAction SilentlyContinue | Where-Object { $_.Name -like '9.*' } | Select-Object -First 1
      if ($hit) { return $true }
    }
    # Fallback via command if available
    $dotnet = Get-DotNetExePath
    if ($dotnet) {
      $rts = Run-Native "`"$dotnet`"" '--list-runtimes'
      if ($rts -match 'Microsoft\.WindowsDesktop\.App\s+9\.') { return $true }
    }
    return $false
  } catch { return $false }
}

function Has-SikuliX205 {
  try {
    if (Test-Path $SikuliJar) {
      $fi = Get-Item $SikuliJar -ErrorAction SilentlyContinue
      if ($fi -and $fi.Length -gt 0) { return $true }
    }
    return $false
  } catch { return $false }
}

function Has-HunterUI {
  try {
    if (Test-Path $HunterUIExe) {
      $fi = Get-Item $HunterUIExe -ErrorAction SilentlyContinue
      if ($fi -and $fi.Length -gt 0) { return $true }
    }
    return $false
  } catch { return $false }
}

function Ensure-PayloadExists {
  if (-not (Test-Path $Payload)) { throw "Payload-map ontbreekt: $Payload" }
}

function Log-PayloadList {
  if (Test-Path $Payload) {
    Write-Log "Payload-inhoud:"
    try {
      Get-ChildItem -Path $Payload | Sort-Object Name | ForEach-Object { Write-Log ("  - " + $_.Name) }
    } catch { Write-Log ("  [kon payload niet lezen] " + $_.Exception.Message) }
  } else {
    Write-Log "Payload-map ontbreekt ( $Payload )."
  }
}

function Find-Temurin17 {
  $patterns = @(
    'OpenJDK17U-jdk_x64_windows_hotspot_*.msi',
    'OpenJDK17*-jdk*_windows_hotspot_*.msi',
    '*Temurin*17*.msi',
    '*jdk*17*.msi'
  )
  foreach ($p in $patterns) {
    $f = Get-ChildItem -Path $Payload -Filter $p -ErrorAction SilentlyContinue | Select-Object -First 1
    if ($f) { return $f }
  }
  return $null
}
function Find-DesktopRt9     { Get-ChildItem -Path $Payload -Filter 'windowsdesktop-runtime-9.*-win-x64.exe' -ErrorAction SilentlyContinue | Select-Object -First 1 }
function Find-SikuliJar205   {
  $f = Get-ChildItem -Path $Payload -Filter 'sikulixapi-2.0.5.jar'   -ErrorAction SilentlyContinue | Select-Object -First 1
  if ($f) { return $f }
  $f = Get-ChildItem -Path $Payload -Filter 'sikulixapi-2.0.5-*.jar' -ErrorAction SilentlyContinue | Select-Object -First 1
  return $f
}

function Install-MSI($msiPath) {
  Write-Log ("MSI: " + (Split-Path $msiPath -Leaf))
  $args = @('/i', "`"$msiPath`"", '/qn', '/norestart')
  $p = Start-Process msiexec.exe -ArgumentList $args -Wait -PassThru -WindowStyle Hidden
  if ($p.ExitCode -ne 0) { throw ("MSI exitcode " + $p.ExitCode) }
}
function Install-EXE($exePath) {
  Write-Log ("EXE: " + (Split-Path $exePath -Leaf))
  $args = @('/quiet','/norestart')
  $p = Start-Process "`"$exePath`"" -ArgumentList $args -Wait -PassThru -WindowStyle Hidden
  if ($p.ExitCode -ne 0) { throw ("EXE exitcode " + $p.ExitCode) }
}
function Install-SikuliX {
  New-Item -ItemType Directory -Force -Path $SikuliDir | Out-Null
  $jar = Find-SikuliJar205
  if (-not $jar) { Write-Log "SikuliX API 2.0.5 JAR niet gevonden in payload."; Log-PayloadList; throw "sikulixapi-2.0.5*.jar ontbreekt." }
  Copy-Item -Force $jar.FullName $SikuliJar
  Write-Log ("SikuliX API geplaatst: " + $jar.Name + " -> " + $SikuliJar)
}

function Install-HunterUI {
  try {
    Write-Log "Downloaden HunterUI-latest.zip van https://csar.bio/HunterUI-latest.zip"
    
    # Download de ZIP file
    $zipPath = Join-Path $env:TEMP "HunterUI-latest.zip"
    $webClient = New-Object System.Net.WebClient
    $webClient.DownloadFile("https://csar.bio/HunterUI-latest.zip", $zipPath)
    Write-Log "Download voltooid: $zipPath"
    
    # Maak HunterUI directory
    New-Item -ItemType Directory -Force -Path $HunterUIDir | Out-Null
    
    # Pak de ZIP uit
    Write-Log "Uitpakken naar: $HunterUIDir"
    Add-Type -AssemblyName System.IO.Compression.FileSystem
    [System.IO.Compression.ZipFile]::ExtractToDirectory($zipPath, $HunterUIDir)
    
    # Verwijder de ZIP file
    Remove-Item $zipPath -Force -ErrorAction SilentlyContinue
    
    Write-Log "HunterUI succesvol geïnstalleerd in: $HunterUIDir"
    
    # Maak desktop shortcut
    Create-DesktopShortcut
    
  } catch {
    Write-Log "Fout bij installatie HunterUI: $($_.Exception.Message)"
    throw
  }
}

function Create-DesktopShortcut {
  try {
    $desktopPath = [Environment]::GetFolderPath("Desktop")
    $shortcutPath = Join-Path $desktopPath "HunterUI.lnk"
    
    Write-Log "Maken desktop shortcut: $shortcutPath"
    
    $WshShell = New-Object -comObject WScript.Shell
    $Shortcut = $WshShell.CreateShortcut($shortcutPath)
    $Shortcut.TargetPath = $HunterUIExe
    $Shortcut.WorkingDirectory = $HunterUIDir
    $Shortcut.Description = "HunterUI Application"
    $Shortcut.Save()
    
    Write-Log "Desktop shortcut gemaakt: $shortcutPath"
  } catch {
    Write-Log "Fout bij maken desktop shortcut: $($_.Exception.Message)"
    # Niet stoppen bij shortcut fout, installatie is al gelukt
  }
}

function Do-Detect {
  try {
    $o = New-Object psobject -Property ([ordered]@{
      Java17     = $false
      DotNet     = $false
      DesktopRt9 = $false
      SikuliX205 = $false
      HunterUI   = $false
    })
    $o.Java17     = Has-Java17
    $o.DotNet     = Has-DotNet
    $o.DesktopRt9 = Has-DesktopRt9
    $o.SikuliX205 = Has-SikuliX205
    $o.HunterUI   = Has-HunterUI
    return $o
  } catch {
    Write-Log ("[detectie] fout genegeerd: " + $_.Exception.Message)
    return (New-Object psobject -Property ([ordered]@{
      Java17=$false; DotNet=$false; DesktopRt9=$false; SikuliX205=$false; HunterUI=$false
    }))
  }
}

function Do-Install([bool]$doTemu, [bool]$doDesk, [bool]$doSik, [bool]$doHunterUI) {
  Ensure-PayloadExists
  New-Item -ItemType Directory -Force -Path $RootDir | Out-Null

  if ($doTemu) {
    $msi = Find-Temurin17
    if (-not $msi) { Write-Log "Temurin JDK 17 MSI niet gevonden in payload."; Log-PayloadList; throw "Temurin 17 MSI ontbreekt." }
    Install-MSI $msi.FullName
    Write-Log "✔ Temurin JDK 17 geïnstalleerd."
  } else {
    Write-Log "Temurin JDK 17 overslaan."
  }


  if ($doDesk) {
    $drt = Find-DesktopRt9
    if (-not $drt) { Write-Log "Windows Desktop Runtime 9 installer niet gevonden in payload."; Log-PayloadList; throw "Desktop Runtime 9 ontbreekt." }
    Install-EXE $drt.FullName
    Write-Log "✔ .NET 9 Desktop Runtime geïnstalleerd."
  } else {
    Write-Log ".NET 9 Desktop Runtime overslaan."
  }

  if ($doSik) {
    Install-SikuliX
    Write-Log "✔ SikuliX API 2.0.5 geplaatst."
  } else {
    Write-Log "SikuliX API overslaan."
  }

  if ($doHunterUI) {
    Install-HunterUI
    Write-Log "✔ HunterUI gedownload en geïnstalleerd."
  } else {
    Write-Log "HunterUI overslaan."
  }
}

# ---------- Start ----------
Ensure-Admin
Write-Log ("Logbestand: " + $LogFile)

if ($Headless) {
  Write-Log "Headless modus – detecteer en installeer ontbrekende onderdelen."
  $d = Do-Detect
  Write-Log ("Detectie: Java17=" + $d.Java17 + " Desktop9=" + $d.DesktopRt9 + " SikuliX205=" + $d.SikuliX205 + " HunterUI=" + $d.HunterUI)

  $doTemu = -not $d.Java17
  $doDesk = -not $d.DesktopRt9
  $doSik  = -not $d.SikuliX205
  $doHunterUI = -not $d.HunterUI

  try {
    Do-Install $doTemu $doDesk $doSik $doHunterUI
    $d2 = Do-Detect
    Write-Log ("Na installatie: Java17=" + $d2.Java17 + " Desktop9=" + $d2.DesktopRt9 + " SikuliX205=" + $d2.SikuliX205 + " HunterUI=" + $d2.HunterUI)
    Write-Log "Klaar."
    exit 0
  } catch {
    Write-Log ("✖ Fout: " + $_.Exception.Message)
    exit 1
  }
}

# ---------- GUI ----------
Add-Type -AssemblyName System.Windows.Forms
Add-Type -AssemblyName System.Drawing

$form = New-Object System.Windows.Forms.Form
$form.Text = "HunterUI Offline Installer"
$form.StartPosition = "CenterScreen"
$form.Size = New-Object System.Drawing.Size(700,520)
$form.MaximizeBox = $false

$lbl = New-Object System.Windows.Forms.Label
$lbl.Text = "Offline installatie (silent). Selecteer wat je wilt installeren:"
$lbl.AutoSize = $true
$lbl.Location = New-Object System.Drawing.Point(20,20)
$form.Controls.Add($lbl)

$chkTemu = New-Object System.Windows.Forms.CheckBox
$chkTemu.Text = "Temurin JDK 17 (LTS)"
$chkTemu.AutoSize = $true
$chkTemu.Location = New-Object System.Drawing.Point(40,60)
$form.Controls.Add($chkTemu)


$chkDesk = New-Object System.Windows.Forms.CheckBox
$chkDesk.Text = ".NET 9 Windows Desktop Runtime (x64)"
$chkDesk.AutoSize = $true
$chkDesk.Location = New-Object System.Drawing.Point(40,90)
$form.Controls.Add($chkDesk)

$chkSik = New-Object System.Windows.Forms.CheckBox
$chkSik.Text = "SikuliX API 2.0.5 → %ProgramData%\HunterUI\sikulix\"
$chkSik.AutoSize = $true
$chkSik.Location = New-Object System.Drawing.Point(40,120)
$form.Controls.Add($chkSik)

$chkHunterUI = New-Object System.Windows.Forms.CheckBox
$chkHunterUI.Text = "HunterUI Application (download van https://csar.bio/HunterUI-latest.zip)"
$chkHunterUI.AutoSize = $true
$chkHunterUI.Location = New-Object System.Drawing.Point(40,150)
$form.Controls.Add($chkHunterUI)

$btnDetect = New-Object System.Windows.Forms.Button
$btnDetect.Text = "Detecteer"
$btnDetect.Location = New-Object System.Drawing.Point(40,190)
$form.Controls.Add($btnDetect)

$btnInstall = New-Object System.Windows.Forms.Button
$btnInstall.Text = "Installeren"
$btnInstall.Location = New-Object System.Drawing.Point(140,190)
$form.Controls.Add($btnInstall)

$txt = New-Object System.Windows.Forms.TextBox
$txt.Multiline = $true
$txt.ScrollBars = "Vertical"
$txt.ReadOnly = $true
$txt.Location = New-Object System.Drawing.Point(40,230)
$txt.Size = New-Object System.Drawing.Size(620,210)
$form.Controls.Add($txt)

$lblLog = New-Object System.Windows.Forms.Label
$lblLog.Text = "Log: $LogFile"
$lblLog.AutoSize = $true
$lblLog.Location = New-Object System.Drawing.Point(40,450)
$form.Controls.Add($lblLog)

function Show-Out($m) { $txt.AppendText($m + "`r`n"); Write-Log $m }

function GUI-Detect {
  Show-Out "Detectie gestart..."
  $d = Do-Detect
  if ($d.Java17) { $chkTemu.Checked = $false } else { $chkTemu.Checked = $true }
  if ($d.DesktopRt9) { $chkDesk.Checked = $false } else { $chkDesk.Checked = $true }
  if ($d.SikuliX205) { $chkSik.Checked = $false } else { $chkSik.Checked = $true }
  if ($d.HunterUI) { $chkHunterUI.Checked = $false } else { $chkHunterUI.Checked = $true }

  Show-Out ("- Temurin JDK 17: " + ($(if ($d.Java17) { "aanwezig" } else { "ontbreekt" })))
  if ($d.DotNet -or (Get-DotNetExePath)) {
    Show-Out ("- .NET 9 Desktop Runtime: " + ($(if ($d.DesktopRt9) { "aanwezig" } else { "ontbreekt" })))
  } else {
    Show-Out "- dotnet.exe niet aangetroffen op vaste locatie."
  }
  Show-Out ("- SikuliX API 2.0.5: " + ($(if ($d.SikuliX205) { "aanwezig" } else { "ontbreekt" })))
  Show-Out ("- HunterUI Application: " + ($(if ($d.HunterUI) { "aanwezig" } else { "ontbreekt" })))

  if (-not (Test-Path $Payload)) { Show-Out ("⚠ Payload-map ontbreekt: " + $Payload) }
  if (-not (Find-Temurin17))     { Show-Out "⚠ Temurin 17 MSI niet gevonden in payload." }
  if (-not (Find-DesktopRt9))    { Show-Out "⚠ Desktop Runtime 9 installer niet gevonden in payload." }
  if (-not (Find-SikuliJar205))  { Show-Out "⚠ sikulixapi-2.0.5*.jar ontbreekt in payload." }
  Show-Out "ℹ HunterUI wordt altijd gedownload van https://csar.bio/HunterUI-latest.zip"
}

$btnDetect.Add_Click({ GUI-Detect })

$btnInstall.Add_Click({
  try {
    Show-Out "Installatie gestart..."
    Do-Install $chkTemu.Checked $chkDesk.Checked $chkSik.Checked $chkHunterUI.Checked
    Show-Out "Her-detectie (PATH-loos):"
    $d2 = Do-Detect
    Show-Out ("Temurin JDK 17: " + ($(if ($d2.Java17) { "aanwezig" } else { "ONTBREEKT" })))
    Show-Out (".NET 9 Desktop RT: " + ($(if ($d2.DesktopRt9) { "aanwezig" } else { "ONTBREEKT" })))
    Show-Out ("SikuliX API 2.0.5: " + ($(if ($d2.SikuliX205) { "aanwezig" } else { "ONTBREEKT" })))
    Show-Out ("HunterUI Application: " + ($(if ($d2.HunterUI) { "aanwezig" } else { "ONTBREEKT" })))
    [System.Windows.Forms.MessageBox]::Show("Installatie afgerond.`nZie log:`n$LogFile","Gereed",'OK','Information') | Out-Null
  } catch {
    Show-Out ("✖ Fout: " + $_.Exception.Message)
    [System.Windows.Forms.MessageBox]::Show("Fout:`n$($_.Exception.Message)`nZie log:`n$LogFile","Fout",'OK','Error') | Out-Null
  }
})

$form.Add_Shown({ GUI-Detect })
[void]$form.ShowDialog()
